/**
* Script: utility.js
* Written by: Radnen
* Updated: 1/15/2011
**/

/**
* Global Utility Object:
* = Contains array and object manipulation functions.
**/
var Utility = ({
	/**
	* foreach(array, func [, parent]);
	* = emulates the C# style of foreach that takes an array
	*   and allows you the use of its elements.
	* = returns: whatever breaks out of the func via a return.
	* - array: the array to iterate through, bottom-up.
	* - func: the function that takes 2 arguments:
	* 	- arg1: the item in question.
	* 	- arg2: reference to parent scope.
	* - parent: the parent scope to be passed to 'func'.
	**/
	foreach: function(array, func, parent) {
		var i = -1, l = array.length - 1, b = false;
		while (i++ < l) if (b = func(array[i], parent)) break;
		return b;
	},
	
	/**
	* repeatFunc(func, times);
	*  - will repeat a function call 'times' times.
	**/
	repeatFunc: function(func, times) {
		var i = times;
		while(i--) func();
	},
	
	/**
	* call(array, call)
	* - array: the array to iterate through top-down.
	* - call: method identifier in object-element of the array.
	* = ex: Utility.call(my_array, "update");
	*   my_array must contain an object with a method called "update".
	**/
	call: function(array, call) {
		var i = array.length;
		while(i--) if (array[i]) array[i][call]();
	},
	
	/**
	* indexOf(array, item);
	*  - array: the array to find an item from top-down.
	*  - item: the object to compare by.
	**/
	indexOf: function(array, item) {
		var i = array.length;
		while(i--) if (array[i] == item) return i;
		return -1;
	},
	
	/**
	* propertyOf(object, item);
	*  = returns the property name of anything in the object that matches the item.
	**/
	propertyOf: function(object, item) {
		for (var i in object) {
			if (object[i] == item) return i;
		}
	},
	
	/**
	* findIn(array, item, prop);
	*  - returns: an object if it exists in the array, by comparing a specific property.
	*  - array: can be a JS 'array' or 'object' object.
	*  = if you gave it an array but no properties, and only an item then it assumes you
	*		 are looking for an index and call Utility.indexOf() instead and return its value.
	*  = conversely, if you gave it an object it will perform Utility.propertyOf() instead.
	**/
	findIn: function(array, item, prop) {
		if (array.length != undefined) {
			if (prop != undefined) {
				var i = array.length;
				while(i--) if (array[i][prop] == item) return array[i];
			}
			else return this.indexOf(array, item);
		}
		else {
			if (prop != undefined) {
				for (var i in array) if (array[i][prop] == item) return array[i];
			}
			else return this.propertyOf(array, item);
		}
		return null;
	},	
	
	/**
	* contains(array, item);
	* - array: the array to search [using foreach].
	* - item: the item to compare against.
	**/
	contains: function(array, item) {
		return this.foreach(array, function(i) { if (i == item) return true; });
	},
	
	/**
	* remove(array, item [, prop]);
	* - array: the array to search top-down.
	* - item: the item to compare by.
	* - prop: additionally, setting the prop will search by a property
	*         in the array if it's an array of objects.
	*         this is useful for removing objects with a name or key.
	* - returns: the item when found.
	**/
	remove: function(array, item, prop) {
		var i = array.length;
		if (prop === undefined) {
			while(i--) { if (array[i] == item) { return array.splice(i, 1); } }
		}
		else {
			while(i--) { if (array[i][prop] == item) { return array.splice(i, 1); } }
		}
	},
	
	/**
	* isSphereObject(obj);
	*  = checks and returns true if your object is a sphere object.
	**/
	isSphereObject: function(obj) {
		if (!obj) return false;
		var str = obj.toString();
		switch(str) {
			case "[object image]":
			case "[object sound]":
			case "[object font]":
			case "[object color]":
			case "[object windowstyle]":
				return true;
			break;
		}
		return false;
	},
	
	/**
	* clearKeys();
	*  = clears the key buffer to unjam it from repeated key presses.
	**/
	clearKeys: function() {
		while(AreKeysLeft()) GetKey();
	},
	
	/**
	* alert(txt);
	*  = used like Abort, except it doesn't quit the game nor show error line number.
	**/
	alert: function(txt) {
		this.clearKeys();
		Rectangle(0, 0, SW, SH, Colors.black);
		GetSystemFont().drawTextBox(0, 0, SW, SH, 0, txt);
		FlipScreen();
		GetKey();
	},
	
	/**
	* copyColor(col [, a]);
	*  = copies the color 'col', if a is true it will not copy alpha.
	**/
	copyColor: function(col, a) {
		if (a) return CreateColor(col.red, col.green, col.blue);			
		else return CreateColor(col.red, col.green, col.blue, col.alpha);
	},
	
	/**
	* set(obj1, obj2);
	*  = sets the properties of object1 to those stored in object2,
	*    ignores sphere objects, and will find nested objects.
	**/
	set: function(obj1, obj2, ignore) {
		for (var i in obj2) {
			//this.alert("("+ i + "), 1: " + obj1[i] + ", 2: " + obj2[i]);
			if (this.isSphereObject(obj1[i])) continue;
			if (ignore && this.contains(ignore, i)) continue;
			if (typeof obj1[i] == "function") continue;
			if (obj1[i] instanceof Object) this.set(obj1[i], obj2[i]);
			else obj1[i] = obj2[i];
		}
	},
	
	/**
	* saveObject(object, file, key);
	*  = saves the object as a JSON string at specified key in file.
	**/
	saveObject: function(object, file, key) {
		file.write(key, JSON.stringify(object));
		file.flush();
	},
	
	/**
	* loadObject(object, file, key [, ignore]);
	*  = makes an object created from a JSON string stored at the key in file.
	*  - object: the object to make loaded changes to, ie: what you used in the save method above.
	*  - ignore: choose to ignore this array of properties from the loaded object.
	**/
	loadObject: function(object, file, key, ignore) {		
		var loaded = JSON.parse(file.read(key, "{}"));
		this.set(object, loaded, ignore);
	},
	
	/**
	* getSavedObject(file, key [, prop]);
	*  = returns an object from a JSON-stored value at 'key' in 'file'. 
	*  - prop: if added retrieves the specific property from an object saved at 'key' in 'file'.
	**/
	getSavedObject: function(file, key, prop) {
		 return JSON.parse(file.read(key, "{}"))[prop];
	},
	
	/**
	* random(min [, max]);
	*  - returns: a random number between min and max.
	*  - if max is null: any number 0 <= x <= min.
	**/
	random: function(min, max) {
		if (!max) { max = min; min = 0; }
		return min+Math.floor(Math.random()*(max+1-min));
	},
	
	/**
	* clone(obj);
	*  - returns: a cloned object.
	**/
	clone: function(obj) {
		function a(){};
		a.prototype = obj;
		return new a;
	},
	
	/**
	* toArray(obj [, prop, filter:{prop, value}]);
	*  - returns the object as an array.
	*  - prop, if defined will filter to only this property. Only used for an object of objects.
	*  - filter: {value: n, prop: p}
	*    filter.value is the value obj[filter.prop] will compare against.
	*    it will then add obj[prop] to the array even if "prop" is different from "filter.prop".
	*  = if no filter it will then check and only add any obj[prop] without filtering.
	*  = if no prop it'll just shallow-copy the object into an array data-structure.
	**/
	toArray: function(obj, prop, filter) {
		var array = [];
		if (!prop)
			for (var i in obj) array.push(obj[i]);
		else {
			if (!filter)
				for (var i in obj) array.push(obj[i][prop]);
			else
				for (var i in obj)
					if (obj[i][filter.prop] == filter.value) array.push(obj[i][prop]);
		}
		return array;
	}
});